package com.onesignal;

import android.app.NotificationChannel;
import android.app.NotificationChannelGroup;
import android.app.NotificationManager;
import android.content.Context;
import android.net.Uri;
import android.os.Build;
import androidx.core.app.NotificationChannelCompat;
import com.onesignal.OneSignal;
import com.onesignal.language.LanguageContext;
import com.onesignal.outcomes.OSOutcomeConstants;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.regex.Pattern;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

/* JADX INFO: Access modifiers changed from: package-private */
/* loaded from: classes.dex */
public class NotificationChannelManager {
    private static final String CHANNEL_PREFIX = "OS_";
    private static final String DEFAULT_CHANNEL_ID = "fcm_fallback_notification_channel";
    private static final String RESTORE_CHANNEL_ID = "restored_OS_notifications";
    private static final Pattern hexPattern = Pattern.compile("^([A-Fa-f0-9]{8})$");

    private static int priorityToImportance(int i) {
        if (i > 9) {
            return 5;
        }
        if (i > 7) {
            return 4;
        }
        if (i > 5) {
            return 3;
        }
        if (i > 3) {
            return 2;
        }
        return i > 1 ? 1 : 0;
    }

    NotificationChannelManager() {
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public static String createNotificationChannel(OSNotificationGenerationJob oSNotificationGenerationJob) {
        if (Build.VERSION.SDK_INT < 26) {
            return DEFAULT_CHANNEL_ID;
        }
        Context context = oSNotificationGenerationJob.getContext();
        JSONObject jsonPayload = oSNotificationGenerationJob.getJsonPayload();
        NotificationManager notificationManager = OneSignalNotificationManager.getNotificationManager(context);
        if (oSNotificationGenerationJob.isRestoring()) {
            return createRestoreChannel(notificationManager);
        }
        if (jsonPayload.has("oth_chnl")) {
            String optString = jsonPayload.optString("oth_chnl");
            if (notificationManager.getNotificationChannel(optString) != null) {
                return optString;
            }
        }
        if (!jsonPayload.has("chnl")) {
            return createDefaultChannel(notificationManager);
        }
        try {
            return createChannel(context, notificationManager, jsonPayload);
        } catch (JSONException e) {
            OneSignal.Log(OneSignal.LOG_LEVEL.ERROR, "Could not create notification channel due to JSON payload error!", e);
            return DEFAULT_CHANNEL_ID;
        }
    }

    /* JADX WARN: Can't wrap try/catch for region: R(27:1|(1:3)(1:64)|4|(1:6)|7|(2:9|(21:11|12|(1:14)|15|(4:17|(1:19)|20|21)|25|(1:27)(1:62)|28|(1:32)|33|(1:35)(1:61)|36|(2:38|(1:40)(2:41|(1:45)))|46|(1:48)(1:60)|49|(1:51)|52|53|54|55))|63|12|(0)|15|(0)|25|(0)(0)|28|(2:30|32)|33|(0)(0)|36|(0)|46|(0)(0)|49|(0)|52|53|54|55) */
    /* JADX WARN: Code restructure failed: missing block: B:58:0x0158, code lost:
    
        r7 = move-exception;
     */
    /* JADX WARN: Code restructure failed: missing block: B:59:0x0159, code lost:
    
        r7.printStackTrace();
     */
    /* JADX WARN: Removed duplicated region for block: B:14:0x0071  */
    /* JADX WARN: Removed duplicated region for block: B:17:0x008e  */
    /* JADX WARN: Removed duplicated region for block: B:27:0x00c8  */
    /* JADX WARN: Removed duplicated region for block: B:35:0x00e7  */
    /* JADX WARN: Removed duplicated region for block: B:38:0x00f5  */
    /* JADX WARN: Removed duplicated region for block: B:48:0x0127  */
    /* JADX WARN: Removed duplicated region for block: B:51:0x0136  */
    /* JADX WARN: Removed duplicated region for block: B:60:0x0129  */
    /* JADX WARN: Removed duplicated region for block: B:61:0x00e9  */
    /* JADX WARN: Removed duplicated region for block: B:62:0x00ca  */
    /*
        Code decompiled incorrectly, please refer to instructions dump.
    */
    private static String createChannel(Context context, NotificationManager notificationManager, JSONObject jSONObject) throws JSONException {
        JSONObject jSONObject2;
        JSONObject jSONObject3;
        NotificationChannel notificationChannel;
        long[] parseVibrationPattern;
        Object opt = jSONObject.opt("chnl");
        if (opt instanceof String) {
            jSONObject2 = new JSONObject((String) opt);
        } else {
            jSONObject2 = (JSONObject) opt;
        }
        String str = DEFAULT_CHANNEL_ID;
        String optString = jSONObject2.optString(OSOutcomeConstants.OUTCOME_ID, DEFAULT_CHANNEL_ID);
        if (!optString.equals(NotificationChannelCompat.DEFAULT_CHANNEL_ID)) {
            str = optString;
        }
        if (jSONObject2.has("langs")) {
            JSONObject jSONObject4 = jSONObject2.getJSONObject("langs");
            String language = LanguageContext.getInstance().getLanguage();
            if (jSONObject4.has(language)) {
                jSONObject3 = jSONObject4.optJSONObject(language);
                notificationChannel = new NotificationChannel(str, jSONObject3.optString("nm", "Miscellaneous"), priorityToImportance(jSONObject.optInt("pri", 6)));
                notificationChannel.setDescription(jSONObject3.optString("dscr", null));
                if (jSONObject2.has("grp_id")) {
                    String optString2 = jSONObject2.optString("grp_id");
                    notificationManager.createNotificationChannelGroup(new NotificationChannelGroup(optString2, jSONObject3.optString("grp_nm")));
                    notificationChannel.setGroup(optString2);
                }
                if (jSONObject.has("ledc")) {
                    String optString3 = jSONObject.optString("ledc");
                    if (!hexPattern.matcher(optString3).matches()) {
                        OneSignal.Log(OneSignal.LOG_LEVEL.WARN, "OneSignal LED Color Settings: ARGB Hex value incorrect format (E.g: FF9900FF)");
                        optString3 = "FFFFFFFF";
                    }
                    try {
                        notificationChannel.setLightColor(new BigInteger(optString3, 16).intValue());
                    } catch (Throwable th) {
                        OneSignal.Log(OneSignal.LOG_LEVEL.ERROR, "Couldn't convert ARGB Hex value to BigInteger:", th);
                    }
                }
                notificationChannel.enableLights(jSONObject.optInt("led", 1) != 1);
                if (jSONObject.has("vib_pt") && (parseVibrationPattern = OSUtils.parseVibrationPattern(jSONObject)) != null) {
                    notificationChannel.setVibrationPattern(parseVibrationPattern);
                }
                notificationChannel.enableVibration(jSONObject.optInt("vib", 1) != 1);
                if (jSONObject.has("sound")) {
                    String optString4 = jSONObject.optString("sound", null);
                    Uri soundUri = OSUtils.getSoundUri(context, optString4);
                    if (soundUri != null) {
                        notificationChannel.setSound(soundUri, null);
                    } else if ("null".equals(optString4) || "nil".equals(optString4)) {
                        notificationChannel.setSound(null, null);
                    }
                }
                notificationChannel.setLockscreenVisibility(jSONObject.optInt("vis", 0));
                notificationChannel.setShowBadge(jSONObject.optInt("bdg", 1) != 1);
                notificationChannel.setBypassDnd(jSONObject.optInt("bdnd", 0) == 1);
                OneSignal.onesignalLog(OneSignal.LOG_LEVEL.VERBOSE, "Creating notification channel with channel:\n" + notificationChannel.toString());
                notificationManager.createNotificationChannel(notificationChannel);
                return str;
            }
        }
        jSONObject3 = jSONObject2;
        notificationChannel = new NotificationChannel(str, jSONObject3.optString("nm", "Miscellaneous"), priorityToImportance(jSONObject.optInt("pri", 6)));
        notificationChannel.setDescription(jSONObject3.optString("dscr", null));
        if (jSONObject2.has("grp_id")) {
        }
        if (jSONObject.has("ledc")) {
        }
        notificationChannel.enableLights(jSONObject.optInt("led", 1) != 1);
        if (jSONObject.has("vib_pt")) {
            notificationChannel.setVibrationPattern(parseVibrationPattern);
        }
        notificationChannel.enableVibration(jSONObject.optInt("vib", 1) != 1);
        if (jSONObject.has("sound")) {
        }
        notificationChannel.setLockscreenVisibility(jSONObject.optInt("vis", 0));
        notificationChannel.setShowBadge(jSONObject.optInt("bdg", 1) != 1);
        notificationChannel.setBypassDnd(jSONObject.optInt("bdnd", 0) == 1);
        OneSignal.onesignalLog(OneSignal.LOG_LEVEL.VERBOSE, "Creating notification channel with channel:\n" + notificationChannel.toString());
        notificationManager.createNotificationChannel(notificationChannel);
        return str;
    }

    private static String createDefaultChannel(NotificationManager notificationManager) {
        NotificationChannel notificationChannel = new NotificationChannel(DEFAULT_CHANNEL_ID, "Miscellaneous", 3);
        notificationChannel.enableLights(true);
        notificationChannel.enableVibration(true);
        notificationManager.createNotificationChannel(notificationChannel);
        return DEFAULT_CHANNEL_ID;
    }

    private static String createRestoreChannel(NotificationManager notificationManager) {
        notificationManager.createNotificationChannel(new NotificationChannel(RESTORE_CHANNEL_ID, "Restored", 2));
        return RESTORE_CHANNEL_ID;
    }

    /* JADX INFO: Access modifiers changed from: package-private */
    public static void processChannelList(Context context, JSONArray jSONArray) {
        if (Build.VERSION.SDK_INT < 26 || jSONArray == null || jSONArray.length() == 0) {
            return;
        }
        NotificationManager notificationManager = OneSignalNotificationManager.getNotificationManager(context);
        HashSet hashSet = new HashSet();
        int length = jSONArray.length();
        for (int i = 0; i < length; i++) {
            try {
                hashSet.add(createChannel(context, notificationManager, jSONArray.getJSONObject(i)));
            } catch (JSONException e) {
                OneSignal.Log(OneSignal.LOG_LEVEL.ERROR, "Could not create notification channel due to JSON payload error!", e);
            }
        }
        if (hashSet.isEmpty()) {
            return;
        }
        List<NotificationChannel> arrayList = new ArrayList<>();
        try {
            arrayList = notificationManager.getNotificationChannels();
        } catch (NullPointerException e2) {
            OneSignal.onesignalLog(OneSignal.LOG_LEVEL.ERROR, "Error when trying to delete notification channel: " + e2.getMessage());
        }
        Iterator<NotificationChannel> it = arrayList.iterator();
        while (it.hasNext()) {
            String id = it.next().getId();
            if (id.startsWith(CHANNEL_PREFIX) && !hashSet.contains(id)) {
                notificationManager.deleteNotificationChannel(id);
            }
        }
    }
}
